/*
 * lcdd.c : linux deamon for a LCD panel
 *          Used for communication with ABng system
 *          LCD panel connected via serial port
 *	    Marcin Michalak : mm@ipl.net
 *	    DSRG AGH '99
 ************************************************************
 * Parameters: location of the configuration file
 *	       if none : assume #LCDD_CFILE (see lcdd.h)
 *	       
 */

#include "lcdd.h"

#ifdef TRACEON
 #define TRTEXT fprintf 
 #define TRTEXT2 fprintf 
#else
 #define TRTEXT (void)
 #define TRTEXT2 (void)
#endif



/***************************************** 
 * lcdd : Parameters : none
 *                     configuration_file
 *****************************************/


/************************************************************ 
 * Signal handler : reacts on :
 * SIGHUP  : restart lcdd with (probably updated) config file
 * SIGINT  : close the daemon nicely
 * SIGALRM : set sigcaught flag to 1
 ************************************************************/
void handleSig(int sg)
{

 switch(sg)
  {
   case SIGHUP:
  
   TRTEXT(stderr,"Restarting the daemon\n");
  
   restart=1; /* Setting the restart flag */    
   break;
   case SIGINT:
    finish=1; 
      
      TRTEXT(stderr,"Shutting down the daemon\n");
         
    break;
   case SIGALRM:
   sigcaught=1;
      TRTEXT(stderr,"Timeout : alarm received\n");
   break;
   default:
   
  
  }/* /switch */

} /* /handle */



char *sys_err_str()
{
 static char msgstr[200];
 
 if(errno!=0)
  {
   if (errno > 0 && errno < sys_nerr)
     sprintf(msgstr,"(%s)", sys_errlist[errno]);
      else
       sprintf(msgstr,"(errno = %d)",errno);
  } else 
     {
      msgstr[0]='\0';
     }
return (msgstr);             
} /* /sys_err_str */



my_perror()
{
 char *sys_err_str();
 TRTEXT(stderr," %s\n",sys_err_str());

}/* /my_perror */

err_sys(va_alist)
va_dcl
{
 va_list args;
 char *fmt;
 
 va_start(args);
 if (pname !=NULL)
   TRTEXT(stderr,"%s: ",pname);
   fmt=va_arg(args,char *);
   vfprintf(stderr,fmt,args);
   va_end(args);
   
   my_perror();
   
   exit(1);
}/* /err_sys */



/**********************************************
 * My fatal error handling 
 **********************************************/
void mferror(char *what)
{
 fprintf(stderr,"%s\n",what);
 exit(1);
}/* /mferror */


/**********************************************
 * My non-fatal error - print message to stderr 
 **********************************************/
void merror(char *what)
{
 TRTEXT(stderr,"%s\n",what);
} /* /merror */

/**************************************
 * Get host IP/mask - from ifconfig 
 **************************************/
int getIPmask(char *ipaddr)
{
 char buff[1024];
 char ifname[30];
 char ipmask[15];
 int a=0;
 struct ifconf ifc;
 struct ifreq *ifr;
 
 struct sockaddr addr;
 struct sockaddr nmask;
 unsigned int bitmask;
 
 inet_sock=socket(AF_INET,SOCK_DGRAM,0);
 if (inet_sock==-1)
  {
   mferror("getIPmask : socket error");
   return 0;
  }
 skfd=inet_sock;
 
 ifc.ifc_len = sizeof(buff);
 ifc.ifc_buf = buff;
 
 /* Try eth0 as default */
 strcpy(ifname,LCDD_IF);
 //??? Co tu zrobic ???
 if (ioctl(skfd,SIOCGIFCONF,&ifc) < 0)
  {
   merror ("getIPmask : ioctl failed");
   return 0;
  }
 ifr = ifc.ifc_req;
 //strcpy(ifname,ifr->ifr_name); //???
 strcpy(ifr->ifr_name, ifname);
 
 if (ioctl(inet_sock, SIOCGIFADDR, ifr) < 0) 
   {
     memset(&addr, 0, sizeof(struct sockaddr));
     mferror("getIPmask : Error getting IP address");
   } 
    else addr = ifr->ifr_addr;
         
 strcpy(ifr->ifr_name, ifname);
 if (ioctl(inet_sock, SIOCGIFNETMASK, ifr) < 0) 
  {
   memset(&nmask, 0, sizeof(struct sockaddr));
  } 
   else nmask = ifr->ifr_netmask;
	       
 

bitmask=(unsigned int)((unsigned int)nmask.sa_data[2] << 24 ) | (unsigned int)(nmask.sa_data[3] << 16 ) | (unsigned int)(nmask.sa_data[4]<< 8) | (unsigned int)nmask.sa_data[5];

a=0;
while (!(bitmask & 1))
 {
  a++;
  bitmask>>=1;
 }
sprintf(ipaddr,"%03u.%03u.%03u.%03u/%d",(unsigned char)addr.sa_data[2],(unsigned char)addr.sa_data[3],(unsigned char)addr.sa_data[4],(unsigned char)addr.sa_data[5],a);
TRTEXT(stderr,"IP : %s\n",ipaddr);

//strncpy(ipaddr,ipmask,18);
//TRTEXT(stderr,"IP 2: %s\n",ipaddr);
close(skfd);
 return 1;
}/* /getIPmask */ 




/******************************************
 * Get value of the string
 * Returns : parsed value 
 ******************************************/
char *getValue(char *pstring)
{
 char gotValue[40];
 char tempchar;
 char ndomore=1;
 char intext=0; /* Is it value in " ", or not ? */
 char *tempstr;
 char glength=0;
 
 tempstr=gotValue;


 TRTEXT2(stderr,"Got: %s",pstring);
 
/* Parsing :
 * Stop when #
 * Eat everything till =
 * Omit white characters \t, ' '
 * get the value till EOF or \n
 * go back and eat white chars
 * put \0 at the end
 */  
 
 /* Search for = */
 do
  {
   tempchar=*pstring;
   if (tempchar=='=')
    {
     intext=1;
    }
   pstring++; 
  
  } while (tempchar!='\n' && tempchar !='#' && !intext && tempchar);
 if (intext)
  {
   
   pstring++;
   intext=0; /* Reuse - flag to exit */
   /* Eat \t and ' ' */
   do
    {
     tempchar=*pstring;
     switch (tempchar)
      {
       case ' ':
       case '\t':
       pstring++;
       break;
       default:
       intext=1;
      }/* /switch */
     
    } while (!intext && tempchar !='\n' && tempchar!='#' && tempchar);
    if (intext)
     {
      /* Let's copy the value till the end */
      
      glength=0;
      tempchar=*pstring;
      while (tempchar !='\n' && tempchar!='#' && tempchar)       
       {
        gotValue[glength]=tempchar;
        glength++;
	tempchar=*(++pstring);
       } 
     }/* /if intext */
     else
      {
       /* There's no value after = */
       return NULL;
      }
  
  }/* /if intext */
  else return NULL;
 if (glength)
  {
   /* Let's eat white chars at the end now */
   intext=1;
   while (glength && intext)
    {
     switch(gotValue[glength-1])
      {
       case '\t':
       case  ' ':
       glength--;
       break;
       default:
       intext=0;
      }
    }/* /while */
    if (glength) /* This is obvious, but it's never a bad idea to ensure */
     {
      glength++; /* Add one for null-termination */
      tempstr=(char *)malloc((sizeof(char))*glength);
      if (!tempstr) mferror("getValue : Memory allocation error");
      strncpy(tempstr,gotValue,glength-1);
      tempstr[glength]=0;      
     }
     else /* This should never occur */
      {
       mferror("Something is wrong");
      }
  
  } /* if glength */
  else 
   {
    /* We parsed 0 characters */
    return NULL;
   } 

 TRTEXT2(stderr,"Parsed : %s,%d\n",tempstr,glength);
 
  return tempstr;
} /* /getValue */  
   


/****************************************************
 * Parsing configuration file, getting settings 
 * Returns : number of read messages or 0 
 * Note : IPcommand is put into messages[0].command
 * for convenience 
 ****************************************************/
int getconf(char *configf,
             char **devicename, 
             struct message **messages
	   )
{
 FILE *confile;	    /* Configuration file */
 char a=1; 	    /* Number of messages */
 char tempstr[255]; /* Read line is put here */
 char messageText[40];
 char insection=0;
 int ipcmd=0;
 struct message *msgs;
 
 
 /* Try opening config file */
 confile=fopen(configf,"r");
 if (confile==NULL)
  {
   mferror("Opening config file failed");
   return 0;
  } 

/* Allocate memory for NMAX messages */
msgs=(struct message *)malloc(NMAX*sizeof(struct message));
if (!(msgs)) mferror("getconf : Memory allocation error");
bzero(msgs,NMAX*sizeof(struct message));

fgets(tempstr,254,confile); /* Read the first line */
insection=0;
 while (*tempstr || *tempstr=='#' || *tempstr=='\n')
  {
   TRTEXT2(stderr,"Parsing : %s",tempstr);
            
   if (insection)
    {
     if (!strncmp("text",tempstr,4))
      {
       msgs[a].text=getValue(tempstr+4);
       if (strlen(msgs[a].text)>maxlength) msgs[a].text[maxlength]=0;
       TRTEXT2(stderr,"Message %d.text:%s\n",a,msgs[a].text);
      }
      if (!strncmp("command",tempstr,7))
      {
       msgs[a].command=getValue(tempstr+7);
       TRTEXT2(stderr,"Message %d.command:%s\n",a,msgs[a].command);
      }
      if (!strncmp("</message>",tempstr,10))
      {
       /* If any item is null, discard the message */
       if (msgs[a].text && msgs[a].command) a++;
        else merror ("getconf : Discarding incomplete message");
       if (a==NMAX) 
        {
	 fclose(confile);
	 merror("getconf : Maximum number of messages reached");
	 return a;
	}
       insection=0;
      }     
    }/* /if insection */
    else /* We're not in section, reading other values */
     {
      if (!strncmp("Device",tempstr,6))
      {
       /* Parse further to get device name */
       *devicename=getValue(tempstr+6);    
      }
      else if (!strncmp("Length",tempstr,6))
      {
       sscanf("%d",getValue(tempstr+6),&maxlength);
      }
      else if (!strncmp("<message>",tempstr,9))
      {
       insection=1;
      }
      else if (!strncmp("IPcommand",tempstr,9))
      {
       msgs[0].command=getValue(tempstr+9);
       ipcmd=1;
      }
     
     } /* else = not in section */


  
   if (!fgets(tempstr,254,confile)) break; /* break if EOF or error */
  } /* /while */ 
 if (insection) a++; /* No </message> after <message> before the end of file */

/* Memory realloc */  
 
 if (a<NMAX) realloc(msgs,a*sizeof(struct message)); 
 
 fclose(confile);
 *messages=msgs;

 if (ipcmd) return a;/* We must know the ip set command !!! */
  else return 0;
} /* /getconf */




/*************************************************** 
 * Open serial port : read by 1 character at a time
 * Returns : fd of the opened device, otherwise 0 
 ***************************************************/
int serialOpen(char *devicename)
{
 struct termios t;
 int fd=0;
 int settings=0;


 if ((devicename == (char *)NULL) || (strlen(devicename) == 0))
  {
   return 0;
  }
 /* Opening the port now */ 
 if((fd= open(devicename, O_RDWR | O_NOCTTY)) < 0)
  {
   merror("Opening device failed");
   return 0;
  }
 bzero(&t, sizeof(t));
 t.c_iflag=0;
 t.c_oflag=0;
 t.c_lflag=0;/* Disable special characters, buffer by characters */


 
 /* 
  * Set control mode flags:
  * B1200  = 1200 bps
  * CS8    = 8n1 
  * CREAD  = Enable receiver
  * CLOCAL = Ignore modem control lines
  * (man termios)
  */ 
  
 
 t.c_cflag= B1200 | CS8 | CREAD | CLOCAL;

t.c_cc[VMIN]=1; /* Minimum no. of chars for read to receive */

 t.c_cc[VEOL]=EOM;/* Set my EOL */
 t.c_cc[VEOF]=EOM;  /* Set EOF    */
 t.c_cc[VSTOP]=EOM; /*  Set STOP  */
 
 tcflush(fd, TCIOFLUSH);  /* Flush I/O */
 
 /* Setting the port parameters */
 if (tcsetattr(fd,TCSANOW, &t) < 0 ) 
  {
   merror ("Setting port failed");
   return 0;
  }
 
 return fd;
} /* /serialOpen */



/******************************************
 * Reset the panel - set PANEL_RST line 
 ******************************************/
int resetDevice(int fd)
{
 int settings=0;
 int tempint=0;
 
 tcflush(fd, TCIOFLUSH);/* Flush I/O */
 settings=PANEL_RST;
 if (ioctl(fd,TIOCMSET,&settings))
  {
   merror("resetDevice: ioctl failed");
   return 0;
  } 
 
sleep(PANEL_ST); /* Waiting PANEL_SleepTime seconds for the device to reset */
tempint=settings^PANEL_RST;

 if (ioctl(fd,TIOCMSET,&tempint))
  {
   merror("resetDevice: ioctl failed");
   return 0;
  }
   
 return 1;
} /* /resetDevice */
 

/* Send preformatted string to device */
int sendString(int fd,char *string)
{
 char len;
 char tempstr[60];
 
 len=strlen(string);
 
 len++;  
 if (sprintf(tempstr,"%s%c",string,EOM ) < 0) 
  {
   merror("SendString: conversion error");
   return 0;
  }


 TRTEXT(stderr,"sendString :Sending :%s\n",tempstr);
 

 if (write(fd,tempstr,len) < len) 
  {
   merror ("sendString: Error writing to device");
   return 0;
  }

 TRTEXT(stderr,"sendString :Sent\n");
     
 return 1;
}




/**********************************************************************
 * sendMessage : send text in "", along with a number and type (M/R/A) 
 * Returns :
 * 2 : if message was send succesfully (& confirmed)
 * 1:  if the message was sent succesfully, but not confirmed 
 * 0 : otherwise
 * mtype : 'M' or 'R' : Message or Result
 **********************************************************************/
int sendMessage(int fd, char *message, char which,char mtype, char cfm)
{ 
 char tempchar;
 char len;         /* Length of the message */
 int count=0;
 int i;
 char tempstr[60]; /* Will be keeping messages here */
 char *wherestr;   /* Where am I in the string now ?*/
 int position;	   /* Position in the string */
 char reply[60];   /* Will store confirmation reply here */
 char result=0;
 char *fmessage;   /* Formatted message with \n's */
 /* 
  * Preparing whole message to be sent, as sending many characters
  * to the device is said to be better than sending each one 
  * separately (that's what the FAQ says)
  */

 len=strlen(message);
 
 /* Cut message length if needed */
 if (len>maxlength) 
  {
   len=maxlength;
   message[maxlength]='\0';
  }
 
 /* Allocating space for text and \n's */
 count=(len-1)/linelength;
 position=0;
 if (count && mtype==PANEL_ER )
 {

  fmessage=(char *)malloc((len+count)*sizeof(char));//!!!???
  count=0;
   if (fmessage==NULL) mferror ("sendMessage : malloc error");
   wherestr=fmessage;
   do
    {
     if (*message=='\n') { count++;position=0;}

     if (position==linelength)
      if (count < 2)
       {
        /* Add newline */
	*(wherestr++)='\n';

	count++; 
	position=0;
       }
        else 
	 {
	  TRTEXT(stderr,"Skracam\n");
          *wherestr='\0';
	  break;
	 } 
     
     *(wherestr++)=*(message++);position++;
     
    
    } while (*message && count < 4);
    *wherestr=*message; /* Add finishing \0 */ 
     
   
   } else fmessage=message;/* No need to insert \n's */
     
len=strlen(fmessage);
 if (which!=-1) 
  {
   if (sprintf(tempstr,"%c%d\"%s\"%c",mtype,which,fmessage,EOM ) < 4)
   {
    merror("sendMessage : sprintf error");
    return result;
   }
  }
  else
   if (sprintf(tempstr,"%c\"%s\"%c",mtype,fmessage,EOM ) < 3)
   {
   merror("sendMessage : sprintf error");
   return result;
   }

 TRTEXT(stderr,"sendMessage :sending %d: %s\n",strlen(tempstr),tempstr);

 len+=5; /* Adding mtype,which,"",EOM */
 if (write(fd,tempstr,len) < len) 
  { 
   merror ("sendMessage: Error writing to device"); 
   return result;
  }
 else result++;


 TRTEXT(stderr,"sendMessage : sent. Checking confirmation\n");

 
 /* Now check the confirmation */
 
if (cfm)
  { 
   receiveString(fd,reply);
   if (!strcmp(reply,tempstr)) result++; /* The result should be now 2 if OK */ 
  }  


 if (result==2) TRTEXT(stderr,"sendMessage : sent & confirmed\n");


  return result;
} /* /sendmessage no. which */


/*******************************************************
 * Receive reply or request from the device 
 * Returns : no. of characters read (result of read())
 *******************************************************/
int receiveString(int fd,char *reply)
{
 char tempchar=0;
 char rlength=0; /* Counted length of the reply */
 char tempstr[60];
 
 reply[0]=0;
 rlength=0;

 /* Read characters as they come one by one */
 do
  {
   read(fd,&tempchar,1);
   reply[rlength++]=tempchar;
  } while (tempchar!=EOM);

  
  tcflush(fd,TCIFLUSH); 

  if (rlength < 1) 
    { 
     merror("receiveString :Error reading device data");
     return rlength;
    }
  reply[rlength]=0;


 TRTEXT(stderr,"receiveString: Received %d chars: %s\n",rlength,reply);

 return rlength;
} /* /receiveString */


/********************************
 * Start the daemon features 
 ********************************/
daemon_start(int ignsigcld)
{
 register int childpid,nfd;
 
 if (getppid()==1)
  goto out;

/* If not run in the background, create child & finish parent */
if ((childpid=fork()) < 0)
 err_sys("can't fork first child");
else if (childpid > 0)
  exit(0); /* parent process */
/* First child */
if (setpgrp() == -1)
   err_sys("can't change process group");

if((childpid=fork()) < 0)
   err_sys("can't fork second child");
 else if (childpid > 0)
      exit(0); /* First child */

out :

errno=0;

//chdir("/");??? Czy zmieniac, czy nie ???
if (ignsigcld) signal(SIGCLD, SIG_IGN);     

TRTEXT(stderr,"5\n");
}/* /daemon_start */

/**********************************
 * Main : does everything else
 **********************************/
void main(int argc,char **argv)
{
 char *configf; 	/* Location of the configuration file */
 struct message *messages;  /* Table with parsed messages, allocated later */
 int nmessages; 	/* Number of messages */
 char *devicename;
 struct hostent *ipaddr2;//Zmienic !!!
 
 FILE *pinput;         /* Input of the pipe command */
 FILE *pidfile;        /* For pid no. */
 int result=0;         /* Result of the called function */
 
 
 int fd=0;              /* fd of the opened port */
 int newpid;            /* For fork */
 
 char tempchar=0;
 char tempstr[60];
 char ipaddr[60];
 static char *tempstr2;
 
 int a;
 int received=0;
 
 tempstr2=(char *)malloc(40*sizeof(char));
 if (!tempstr2) mferror("lcdd : Memory allocation error");

 
#ifndef DEBUG
daemon_start(0);
TRTEXT(stderr,"Daemon started\n");
#endif	    



/* Getting command line parameters */
if (argc > 1) configf=argv[1]; 
 else configf=LCDD_CFILE;		 

/* Get settings from the config file */
nmessages=getconf(configf,
                 &devicename, 
                 &messages
	    );




/* Write message and command, if needed */
for (a=1;a<nmessages;a++)
{
 TRTEXT(stderr,"Message %d:%s; %s\n",a,messages[a].text,messages[a].command);
}
TRTEXT(stderr,"IPcommand :%s\n",messages[0].command);


/* Now test if we're ready to start work 
 * We need : nmessages > 0
 *           serial port opened succesfully
 */

/* Install signal handlers */

signal(SIGHUP,handleSig);
signal(SIGINT,handleSig);
signal(SIGALRM,handleSig);

sigcaught=0;

siginterrupt(SIGINT,1);/* Allow read termination by SIGINT */
siginterrupt(SIGHUP,1);/* Allow read termination by SIGHUP */ 
siginterrupt(SIGALRM,1);/* Allow read termination by SIGALRM */

alarm(LCDD_TIMEOUT); /* Set timer */
if (nmessages) fd=serialOpen(devicename);
 else
  {
   alarm(0);
   mferror("Error in config file. Exiting");
  }
alarm(0);  

if (fd==-1 || sigcaught) 
  {
   mferror("Can't open serial port. Exiting");
  }
 else
  {
   /* OK */
   
  }  

/* Everything is ok so far, let's get to work then */


sigcaught=0;

/* Write pid to LCDD_PIDFILE */
pidfile=fopen(LCDD_PIDFILE,"w");
sprintf(tempstr,"%d\n",getpid());
fwrite(tempstr,sizeof(char),strlen(tempstr),pidfile);
fclose(pidfile);
tempstr[0]=0;


/* First reset the panel */
Reset :
if (restart || finish) goto cleanup;
TRTEXT(stderr,"main : resetting %d time\n",attempts);
sigcaught=0;  


if (attempts<=LCDD_MAXATT || LCDD_CONT)
{
 alarm(LCDD_TIMEOUT);
 if(!resetDevice(fd) || sigcaught) 
  {
   alarm(0);
   goto cleanup;
  }
  else alarm(0);
  
}
else 
 {
  merror("Giving up. Maximum no. of attempts reached"); 
  goto cleanup;
 }


/* Wait for "I"nitialized */
alarm(LCDD_TIMEOUT);
tempstr2[0]=0;
receiveString(fd,tempstr2);
if (strcmp(tempstr2,PANEL_INIT)) 
 {
  attempts++;
  
 TRTEXT(stderr,"main :Didn't receive I\n");

  alarm(0);  
  goto Reset;
 }
 else 
  {
   //attempts=0;
   alarm(0);   /* Cancel timeout */
   TRTEXT(stderr,"main : received I\n");
  }

/* Send current IP/mask to the panel */
SendIP:

   alarm(LCDD_TIMEOUT);
   
/* Get host IP/mask - like in ifconfig */
if (!getIPmask(ipaddr))
 { 
  merror("getIPmask failed");
  alarm(0);
  goto cleanup;
 }
   result=sendMessage(fd,ipaddr,-1,'A',1);
//getchar();
   alarm(0);
   if (result!=2)
    {
     attempts++;
     goto Reset;
    }

/* Send no. of messages to be downloaded */
SendD:
if (restart || finish) goto cleanup;

/* Send info about no. of messages */

sprintf(tempstr,"%c%d",PANEL_LM,nmessages-1);
alarm(LCDD_TIMEOUT);
if (!sendString(fd,tempstr))
 {
  alarm(0);
  goto Reset;
 }
 else
  {
   alarm(0);
  }

SendMessages: /* Let panel receive the messages */
if (restart || finish) goto cleanup;
 for (a=1;a<nmessages;a++)
  {
   if (restart || finish) goto cleanup;
   /* Receive request for message a */
   sprintf(tempstr,"%c%d",PANEL_RM,a);
   alarm(LCDD_TIMEOUT);
   tempstr2[0]=0;
   receiveString(fd,tempstr2);
   alarm(0);/* Cancel timeout */
   if (strcmp(tempstr2,tempstr)) 
    {
     
     attempts++;
     
 TRTEXT(stderr,"main :Didn't receive R%d\n",a);
     goto Reset;
   }/* /if strcmp */
 else /* OK - received request */
  {
   attempts=0;
   
   TRTEXT(stderr,"Received R%d\n",a);
  }/* /else */

  /* Send requested message */
  alarm(LCDD_TIMEOUT);
  if (sendMessage(fd,messages[a].text,a,PANEL_SM,1)!=2)
   {
    /* Text sent unsuccesfully */
    alarm(0);
    break; /* Exit for */
   }
   else
   {
    /* OK */
    alarm(0);
    TRTEXT(stderr,"Sent %s\n",messages[a].text);
   }     
 }/* /for  : finished sending */ 

 if (a!=nmessages) 
  {
   /* Something went wrong - let's start again */
   attempts++;
   goto Reset;
  }  

ReceiveReady:
 if (restart || finish) goto cleanup;
 alarm(LCDD_TIMEOUT);
 tempstr2[0]=0;
 receiveString(fd,tempstr2);
 alarm(0); 
 if (strcmp(tempstr2,PANEL_BG)) 
    {
     attempts++;
  TRTEXT(stderr,"Didn't receive B\n");
     goto Reset;
    }/* /if strcmp */
    else /* OK */
    {
     attempts=0;
     TRTEXT(stderr,"Received B\n");
    }/* /else */

WaitForCommand:
 if (restart || finish) goto cleanup;
 sigcaught=0;
 alarm(2*PANEL_TIMEOUT);/* Loop for NOP or Execution request */
 tempstr2[0]=0;
 receiveString(fd,tempstr2);
 alarm(0);
 if (sigcaught==1) 
  {
   /* Nothing received during 2*PANEL_NOP seconds */
    TRTEXT(stderr,"Neither NOP nor command received. Resetting\n");
   
   goto Reset;
  } 
  else
   {
    
    if (!strcmp(tempstr2,PANEL_NOP))
     {
      /* NOP received - start waiting again */  
        TRTEXT(stderr,"Received NOP\n");
      //??? Dopisac alarm !!! ???
      alarm(LCDD_TIMEOUT);
      result=sendString(fd,PANEL_NOP);/* Reply with NOP */
      alarm(0);
      if (!result) goto Reset;//???
      goto WaitForCommand;
     }
     else
     {
      /* Received something else then NOP - try to parse it now */
        TRTEXT(stderr,"Received something else than NOP\n");
      ipaddr[0]=0;
      tempstr[0]=0;
      sprintf(tempstr,"%s%%d\"%%s",PANEL_EX);//!!! Czy dziala !!!
      if (strlen(tempstr2)>1) result=sscanf(tempstr2,tempstr,&a,ipaddr); //PANEL_EX="E"
       else
        {
         merror("Reply too short");
         goto Reset;
        }

      if (strlen(ipaddr)>0)
       if (ipaddr[strlen(ipaddr)-1]=='\"') ipaddr[strlen(ipaddr)-1]=0;/* Eat remaining " */

      switch(result)
       {
       case 1: /* Execute command */
       case 2:
        TRTEXT(stderr,"Executing %d\n",a);
       if (a >-1 && a < nmessages) goto ExecuteCommand;
        else {
              goto Reset;
	     }
       break;
	
       default:
              goto Reset;
      }/* /switch */
     }/* /else */    
   }/* /else = someting received */

ExecuteCommand: /* Execute requested command */
 if (restart || finish) goto cleanup;
 TRTEXT(stderr,"Executing the command\n");
 alarm(2*LCDD_TIMEOUT);/* Start the timer */
  if (*ipaddr)
   {
    sprintf(tempstr,"%s %s",messages[a].command,ipaddr);
   }
   else
    {
     sprintf(tempstr,"%s",messages[a].command);
    }
  pinput=popen(tempstr,"r");
  if (pinput!=NULL)
     {
      tempstr2[0]=0;
      received=0;
      received=fread(tempstr2,1,40,pinput);
      pclose(pinput);
      alarm(0);
      if (received>0)
       {
        /* Either received the signal or execution error */
        if (tempstr2[received-1]=='\n') tempstr2[received-1]=0; /* Eat finishing \n */
        TRTEXT(stderr,"After execution :received %s\n",tempstr2);
       }/* /if received */	
     }/* /if pinput */
  else
   {
      /* Pipe not created */
       alarm(0);
       close(fd);
       merror("Pipe not created - exiting");
       goto cleanup;
      }  
    
SendResult: /* Send execution result */
TRTEXT(stderr,"send Result\n");
 if (restart || finish) goto cleanup;
 alarm(LCDD_TIMEOUT);
 if (!tempstr2[0]) strcpy(tempstr2,"No result");//???
 if (!sendMessage(fd,tempstr2,a,PANEL_ER,1)) 
  {
   /* An error has occured */
   alarm(0);
   attempts++;
   goto Reset;
  }
  else
   {
    /* OK */
    alarm(0);
  TRTEXT(stderr,"Sent result: %s\n",tempstr2);
    goto WaitForCommand;
   }     

cleanup:
 close(fd);
 unlink(LCDD_PIDFILE);/* Delete the file with pid */
 
 /* Restart, if needed */
 if (restart)
 {
   TRTEXT(stderr,"Restarting\n");
  execve(argv[0],argv,NULL);
  exit(0); /* We should never be here */
 } 

} /* /main */

